<?php

namespace App\Http\Controllers;

use App\Models\Juego;
use App\Models\Carton;
use App\Models\User;
use App\Events\BolaSorteada;
use App\Events\MensajeAdmin;
use App\Events\SorteoAutoIniciado;
use App\Events\SorteoAutoDetenido;
use App\Events\JuegoReiniciado;
use App\Events\ConfiguracionActualizada;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class AdminController extends Controller
{
    const ADMIN_PASSWORD = 'bingo2024';

    public function loginForm()
    {
        if (Session::get('is_admin')) {
            return redirect()->route('admin.panel');
        }
        return view('admin.login');
    }

    public function login(Request $request)
    {
        $request->validate([
            'password' => 'required|string',
        ]);

        if ($request->password === self::ADMIN_PASSWORD) {
            Session::put('is_admin', true);
            Session::put('nombre_jugador', '🎰 MODERADOR');
            return redirect()->route('admin.panel')->with('success', '¡Bienvenido Moderador!');
        }

        return back()->with('error', 'Contraseña incorrecta');
    }

    public function logout()
    {
        Session::forget('is_admin');
        return redirect()->route('bingo.index')->with('success', 'Sesión de moderador cerrada');
    }

    public function panel()
    {
        if (!Session::get('is_admin')) {
            return redirect()->route('admin.login');
        }

        $juego = Juego::whereIn('estado', ['esperando', 'en_curso'])->first();
        
        if (!$juego) {
            $juego = Juego::create([
                'precio_carton' => 1000,
                'acumulado' => 50000,
            ]);
        }

        $totalJugadores = $juego->cartones()->distinct('session_id')->count('session_id');
        $totalCartones = $juego->cartones()->count();
        $bolasSorteadas = $juego->bolasSorteadas()->count();
        
        $jugadores = $juego->cartones()
            ->select('session_id')
            ->selectRaw('COUNT(*) as total_cartones')
            ->selectRaw('SUM(apuesta) as total_apuestas')
            ->groupBy('session_id')
            ->get()
            ->map(function ($item) {
                $nombre = cache()->get('jugador_nombre_' . $item->session_id, 'Anónimo');
                return [
                    'nombre' => $nombre,
                    'cartones' => $item->total_cartones,
                    'apuestas' => $item->total_apuestas,
                ];
            });

        $historial = $juego->bolasSorteadas()->orderBy('orden', 'desc')->take(10)->get();
        $sorteoActivo = cache()->get('sorteo_auto_' . $juego->id);

        return view('admin.panel', compact(
            'juego',
            'totalJugadores',
            'totalCartones',
            'bolasSorteadas',
            'jugadores',
            'historial',
            'sorteoActivo'
        ));
    }

    public function iniciarSorteo(Juego $juego)
    {
        if (!Session::get('is_admin')) {
            return response()->json(['error' => 'No autorizado'], 403);
        }

        if ($juego->estado === 'finalizado') {
            return response()->json(['error' => 'Juego finalizado'], 400);
        }

        $sessionId = Session::getId();
        
        cache()->put('sorteo_auto_' . $juego->id, [
            'session_id' => $sessionId,
            'nombre' => '🎰 MODERADOR',
            'iniciado_at' => now(),
            'es_admin' => true,
        ], now()->addHours(1));

        if ($juego->estado === 'esperando') {
            $juego->iniciar();
        }

        broadcast(new SorteoAutoIniciado(
            $juego->id,
            $sessionId,
            '🎰 MODERADOR',
            $juego->estado
        ));
        \Log::info("Sorteo auto iniciado en juego.{$juego->id}");

        return response()->json([
            'success' => true,
            'mensaje' => 'Sorteo iniciado por el moderador',
        ]);
    }

    public function detenerSorteo(Juego $juego)
    {
        if (!Session::get('is_admin')) {
            return response()->json(['error' => 'No autorizado'], 403);
        }

        cache()->forget('sorteo_auto_' . $juego->id);

        broadcast(new SorteoAutoDetenido($juego->id, Session::getId()));

        return response()->json([
            'success' => true,
            'mensaje' => 'Sorteo detenido',
        ]);
    }

    public function sortearBola(Juego $juego)
    {
        if (!Session::get('is_admin')) {
            return response()->json(['error' => 'No autorizado'], 403);
        }

        if ($juego->estado === 'finalizado') {
            return response()->json(['error' => 'Juego finalizado'], 400);
        }

        if ($juego->estado === 'esperando') {
            $juego->iniciar();
        }

        $bola = $juego->sortearBola();

        if (!$bola) {
            $juego->finalizar();
            cache()->forget('sorteo_auto_' . $juego->id);
            return response()->json([
                'error' => 'Límite alcanzado',
                'finalizado' => true,
            ], 400);
        }

        $this->actualizarCartones($juego, $bola->numero);
        $ganadorLinea = $this->verificarGanadorLinea($juego);
        $ganadorBingo = $this->verificarGanadorBingo($juego);

        $juego->refresh();
        $juegoFinalizado = $ganadorBingo !== null || $juego->haAlcanzadoLimite();

        if ($juego->haAlcanzadoLimite() && !$ganadorBingo) {
            $juego->finalizar();
            cache()->forget('sorteo_auto_' . $juego->id);
        }

        $bolaData = [
            'numero' => $bola->numero,
            'letra' => $bola->letra,
            'orden' => $bola->orden,
            'color' => $bola->getColorClass(),
            'display' => $bola->getDisplay(),
        ];

        try {
            broadcast(new BolaSorteada(
                $juego->id,
                $bolaData,
                $juego->bolasSorteadas()->count(),
                $juego->getMaxBolas(),
                $ganadorLinea,
                $ganadorBingo,
                $juegoFinalizado,
                $juego->haAlcanzadoLimite(),
                (float) $juego->acumulado,
                (float) $juego->premio_linea,
                (float) $juego->premio_bingo,
                $juego->estado
            ));
            \Log::info("Broadcast enviado: bola {$bola->numero} al canal juego.{$juego->id}");
        } catch (\Exception $e) {
            \Log::error('Error en broadcast: ' . $e->getMessage());
        }

        return response()->json([
            'bola' => $bolaData,
            'bolas_sorteadas' => $juego->bolasSorteadas()->count(),
            'ganador_linea' => $ganadorLinea,
            'ganador_bingo' => $ganadorBingo,
            'finalizado' => $juegoFinalizado,
        ]);
    }

    public function enviarMensaje(Request $request, Juego $juego)
    {
        if (!Session::get('is_admin')) {
            return response()->json(['error' => 'No autorizado'], 403);
        }

        $request->validate([
            'mensaje' => 'required|string|max:200',
            'tipo' => 'required|in:info,alerta,exito,promo',
            'gif' => 'nullable|string|url',
        ]);

        broadcast(new MensajeAdmin(
            $juego->id,
            $request->mensaje,
            $request->tipo,
            $request->gif
        ));

        return response()->json([
            'success' => true,
            'mensaje' => 'Mensaje enviado a todos los jugadores',
        ]);
    }

    public function reiniciarJuego(Juego $juego)
    {
        if (!Session::get('is_admin')) {
            return response()->json(['error' => 'No autorizado'], 403);
        }

        cache()->forget('sorteo_auto_' . $juego->id);
        
        $reembolsos = $this->devolverDineroJugadores($juego);
        
        $juego->bolasSorteadas()->delete();
        $juego->cartones()->delete();
        
        $juego->update([
            'estado' => 'esperando',
            'acumulado' => 50000,
            'premio_linea' => 0,
            'premio_bingo' => 0,
            'bolas_sorteadas' => 0,
            'ganador_linea_id' => null,
            'ganador_bingo_id' => null,
        ]);

        $mensajeReembolso = $reembolsos['total'] > 0 
            ? "🔄 Juego reiniciado. 💰 Se devolvió $" . number_format($reembolsos['total'], 0) . " a {$reembolsos['jugadores']} jugador(es)"
            : '🔄 El moderador ha reiniciado el juego. ¡Compra nuevos cartones!';

        broadcast(new JuegoReiniciado(
            $juego->id,
            $mensajeReembolso
        ));

        Log::info('Juego reiniciado con reembolsos', $reembolsos);

        return response()->json([
            'success' => true,
            'mensaje' => 'Juego reiniciado',
            'reembolsos' => $reembolsos,
        ]);
    }

    private function devolverDineroJugadores(Juego $juego): array
    {
        $totalDevuelto = 0;
        $jugadoresReembolsados = 0;
        $precioCarton = $juego->precio_carton;

        $cartonesAgrupados = $juego->cartones()
            ->select('user_id', 'session_id')
            ->selectRaw('SUM(apuesta) as total_apuestas')
            ->selectRaw('COUNT(*) as cantidad_cartones')
            ->groupBy('user_id', 'session_id')
            ->get();

        foreach ($cartonesAgrupados as $grupo) {
            $montoDevolver = $grupo->total_apuestas * $precioCarton;
            
            if ($montoDevolver <= 0) continue;

            if ($grupo->user_id) {
                $user = User::find($grupo->user_id);
                if ($user) {
                    $user->increment('balance', $montoDevolver);
                    Log::info("Reembolso a usuario {$user->id}: \${$montoDevolver}");
                    $totalDevuelto += $montoDevolver;
                    $jugadoresReembolsados++;
                }
            } else {
                $reembolsoPendiente = cache()->get('reembolso_pendiente_' . $grupo->session_id, 0);
                cache()->put('reembolso_pendiente_' . $grupo->session_id, $reembolsoPendiente + $montoDevolver, now()->addHours(24));
                Log::info("Reembolso pendiente para sesión {$grupo->session_id}: \${$montoDevolver}");
                $totalDevuelto += $montoDevolver;
                $jugadoresReembolsados++;
            }
        }

        return [
            'total' => $totalDevuelto,
            'jugadores' => $jugadoresReembolsados,
        ];
    }

    private function actualizarCartones(Juego $juego, int $numero): void
    {
        foreach ($juego->cartones as $carton) {
            $carton->marcarNumero($numero);
        }
    }

    private function verificarGanadorLinea(Juego $juego)
    {
        if ($juego->ganador_linea_id) {
            return null;
        }

        foreach ($juego->cartones as $carton) {
            if (!$carton->tiene_linea && $carton->verificarLinea()) {
                $carton->update(['tiene_linea' => true]);
                $juego->update(['ganador_linea_id' => $carton->user_id]);
                
                $premioTotal = $juego->premio_linea * $carton->apuesta;
                $nombreGanador = cache()->get('jugador_nombre_' . $carton->session_id, 'Jugador');
                
                return [
                    'carton_id' => $carton->id,
                    'premio' => $premioTotal,
                    'nombre' => $nombreGanador,
                ];
            }
        }
        return null;
    }

    private function verificarGanadorBingo(Juego $juego)
    {
        foreach ($juego->cartones as $carton) {
            if (!$carton->tiene_bingo && $carton->verificarBingo()) {
                $carton->update(['tiene_bingo' => true]);
                $juego->update(['ganador_bingo_id' => $carton->user_id]);
                $juego->finalizar();
                cache()->forget('sorteo_auto_' . $juego->id);
                
                $premioTotal = ($juego->premio_bingo + $juego->acumulado) * $carton->apuesta;
                $nombreGanador = cache()->get('jugador_nombre_' . $carton->session_id, 'Jugador');
                
                return [
                    'carton_id' => $carton->id,
                    'premio' => $premioTotal,
                    'nombre' => $nombreGanador,
                ];
            }
        }
        return null;
    }

    public function cambiarMaxBolas(Request $request, Juego $juego)
    {
        \Log::info('cambiarMaxBolas llamado', ['juego_id' => $juego->id, 'max_bolas' => $request->max_bolas, 'estado' => $juego->estado]);
        
        if (!Session::get('is_admin')) {
            return response()->json(['error' => 'No autorizado'], 403);
        }

        if ($juego->estado === 'en_curso') {
            return response()->json(['error' => 'No puedes cambiar las bolas mientras el sorteo está en curso'], 400);
        }

        $maxBolas = (int) $request->max_bolas;
        
        if (!in_array($maxBolas, Juego::OPCIONES_MAX_BOLAS)) {
            return response()->json(['error' => 'Valor no válido. Usa 50, 60 o 70'], 400);
        }

        $juego->update(['max_bolas' => $maxBolas]);
        
        \Log::info('maxBolas actualizado a ' . $maxBolas);

        broadcast(new ConfiguracionActualizada(
            $juego->id,
            $maxBolas,
            "El moderador cambió el sorteo a {$maxBolas} bolas"
        ));

        return response()->json([
            'success' => true,
            'mensaje' => "Sorteo configurado a {$maxBolas} bolas",
            'max_bolas' => $maxBolas,
        ]);
    }
}
