<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class Carton extends Model
{
    protected $table = 'cartones';

    protected $fillable = [
        'juego_id',
        'user_id',
        'session_id',
        'numeros',
        'marcados',
        'tiene_linea',
        'tiene_bingo',
        'apuesta',
        'posicion',
    ];

    protected $casts = [
        'numeros' => 'array',
        'marcados' => 'array',
        'tiene_linea' => 'boolean',
        'tiene_bingo' => 'boolean',
        'apuesta' => 'decimal:2',
    ];

    public function juego(): BelongsTo
    {
        return $this->belongsTo(Juego::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public static function generarNumeros(): array
    {
        $carton = [];
        $columnas = [
            'B' => range(1, 15),
            'I' => range(16, 30),
            'N' => range(31, 45),
            'G' => range(46, 60),
            'O' => range(61, 75),
        ];

        foreach ($columnas as $letra => $rango) {
            shuffle($rango);
            $carton[$letra] = array_slice($rango, 0, 5);
        }

        $carton['N'][2] = 0;

        return $carton;
    }

    public function marcarNumero(int $numero): bool
    {
        $marcados = $this->marcados ?? [];
        
        foreach ($this->numeros as $letra => $columna) {
            if (in_array($numero, $columna)) {
                $marcados[] = $numero;
                $this->update(['marcados' => array_unique($marcados)]);
                return true;
            }
        }
        
        return false;
    }

    public function verificarLinea(): bool
    {
        $marcados = $this->marcados ?? [];
        
        for ($fila = 0; $fila < 5; $fila++) {
            $lineaCompleta = true;
            foreach ($this->numeros as $letra => $columna) {
                $numero = $columna[$fila];
                if ($numero !== 0 && !in_array($numero, $marcados)) {
                    $lineaCompleta = false;
                    break;
                }
            }
            if ($lineaCompleta) {
                return true;
            }
        }
        
        return false;
    }

    public function verificarBingo(): bool
    {
        $marcados = $this->marcados ?? [];
        
        foreach ($this->numeros as $letra => $columna) {
            foreach ($columna as $numero) {
                if ($numero !== 0 && !in_array($numero, $marcados)) {
                    return false;
                }
            }
        }
        
        return true;
    }

    public function getNumerosMarcadosCount(): int
    {
        return count($this->marcados ?? []);
    }

    public function getNumerosRestantes(): int
    {
        $totalNumeros = 24;
        return $totalNumeros - $this->getNumerosMarcadosCount();
    }
}
