<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class Juego extends Model
{
    public const DEFAULT_MAX_BOLAS = 60;
    public const OPCIONES_MAX_BOLAS = [50, 60, 70];

    protected $fillable = [
        'codigo',
        'estado',
        'precio_carton',
        'acumulado',
        'premio_linea',
        'premio_bingo',
        'bolas_sorteadas',
        'max_bolas',
        'ganador_linea_id',
        'ganador_bingo_id',
        'inicio_at',
        'fin_at',
    ];

    protected $casts = [
        'precio_carton' => 'decimal:2',
        'acumulado' => 'decimal:2',
        'premio_linea' => 'decimal:2',
        'premio_bingo' => 'decimal:2',
        'inicio_at' => 'datetime',
        'fin_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($juego) {
            if (empty($juego->codigo)) {
                $juego->codigo = 'BINGO-' . strtoupper(Str::random(6));
            }
        });
    }

    public function cartones(): HasMany
    {
        return $this->hasMany(Carton::class);
    }

    public function bolasSorteadas(): HasMany
    {
        return $this->hasMany(BolaSorteada::class)->orderBy('orden');
    }

    public function ganadorLinea(): BelongsTo
    {
        return $this->belongsTo(User::class, 'ganador_linea_id');
    }

    public function ganadorBingo(): BelongsTo
    {
        return $this->belongsTo(User::class, 'ganador_bingo_id');
    }

    public function getUltimaBola()
    {
        return $this->bolasSorteadas()->latest('orden')->first();
    }

    public function getNumerosDisponibles(): array
    {
        $sorteados = $this->bolasSorteadas()->pluck('numero')->toArray();
        $todos = range(1, 75);
        return array_diff($todos, $sorteados);
    }

    public function sortearBola(): ?BolaSorteada
    {
        if ($this->bolas_sorteadas >= $this->getMaxBolas()) {
            return null;
        }

        $disponibles = $this->getNumerosDisponibles();
        
        if (empty($disponibles)) {
            return null;
        }

        $numero = $disponibles[array_rand($disponibles)];
        $letra = $this->getLetraParaNumero($numero);

        $bola = $this->bolasSorteadas()->create([
            'numero' => $numero,
            'letra' => $letra,
            'orden' => $this->bolas_sorteadas + 1,
        ]);

        $this->increment('bolas_sorteadas');

        return $bola;
    }

    public function haAlcanzadoLimite(): bool
    {
        return $this->bolas_sorteadas >= $this->getMaxBolas();
    }

    public function getMaxBolas(): int
    {
        return $this->max_bolas ?? self::DEFAULT_MAX_BOLAS;
    }

    public function getLetraParaNumero(int $numero): string
    {
        return match (true) {
            $numero <= 15 => 'B',
            $numero <= 30 => 'I',
            $numero <= 45 => 'N',
            $numero <= 60 => 'G',
            default => 'O',
        };
    }

    public function iniciar(): void
    {
        $this->update([
            'estado' => 'en_curso',
            'inicio_at' => now(),
        ]);
    }

    public function finalizar(): void
    {
        $this->update([
            'estado' => 'finalizado',
            'fin_at' => now(),
        ]);
    }
}
