@extends('layouts.app')

@section('title', 'Jugar Bingo')

@section('content')
<div class="min-h-[calc(100vh-140px)] p-2 md:p-4" x-data="bingoGame()" x-init="init()">
    <div class="max-w-7xl mx-auto space-y-3">

        <!-- Balance del jugador -->
        <div class="glass-effect rounded-xl p-3 border border-cyan-500/30 bg-gradient-to-r from-cyan-500/10 to-blue-500/10">
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-3">
                    <div class="w-10 h-10 rounded-full bg-gradient-to-br from-cyan-400 to-blue-500 flex items-center justify-center shadow-lg shadow-cyan-500/30">
                        <span class="text-xl">💰</span>
                    </div>
                    <div>
                        <p class="text-[10px] text-cyan-400/80 uppercase tracking-wider font-semibold">Tu Saldo</p>
                        <p class="text-2xl font-black text-cyan-400">$<span x-text="Number(balance).toLocaleString()">{{ number_format($balance, 0) }}</span></p>
                    </div>
                </div>
                <div class="flex items-center gap-2">
                    <div x-show="ultimaGanancia" x-transition class="px-3 py-1 rounded-full bg-green-500/20 border border-green-500/30">
                        <span class="text-sm font-bold text-green-400">+$<span x-text="Number(ultimaGanancia).toLocaleString()"></span></span>
                    </div>
                    <a href="{{ route('bingo.recargar') }}" 
                       class="px-3 py-2 rounded-lg text-xs font-bold bg-gradient-to-r from-green-500 to-emerald-600 text-white hover:from-green-400 hover:to-emerald-500 transition-all flex items-center gap-1">
                        <span>💳</span> Recargar
                    </a>
                </div>
            </div>
        </div>

        <!-- Jugadores activos -->
        <div class="glass-effect rounded-xl p-3 border border-indigo-500/30 bg-gradient-to-r from-indigo-500/10 to-purple-500/10">
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-3">
                    <span class="text-xs text-indigo-400 font-semibold uppercase">👤 Jugando como:</span>
                    <span class="px-3 py-1 rounded-full text-sm font-bold bg-yellow-500/20 text-yellow-400 border border-yellow-500/30">
                        {{ $nombreJugador }}
                    </span>
                </div>
                <div class="flex items-center gap-2">
                    <div class="w-2 h-2 rounded-full bg-green-500 animate-pulse"></div>
                    <span class="text-sm font-bold text-green-400" x-text="(jugadores.length + 1) + ' jugador' + (jugadores.length > 0 ? 'es' : '') + ' en línea'"></span>
                </div>
            </div>
        </div>

        <div class="grid grid-cols-3 gap-3 md:gap-6">
            <div class="glass-effect rounded-2xl p-4 md:p-6 text-center border-2 border-green-500/30 bg-gradient-to-br from-green-500/10 to-emerald-500/10">
                <p class="text-sm md:text-base text-green-400 uppercase tracking-wider font-semibold mb-1">🎯 Línea</p>
                <p class="text-3xl md:text-5xl font-black text-green-400">
                    $<span x-text="Number(premioLinea).toLocaleString()">{{ number_format($juego->premio_linea ?? 0, 0) }}</span>
                </p>
            </div>
            <div class="glass-effect rounded-2xl p-4 md:p-6 text-center border-2 border-yellow-500/50 bg-gradient-to-br from-yellow-500/20 to-orange-500/20 shadow-lg shadow-yellow-500/20">
                <p class="text-sm md:text-base text-yellow-400 uppercase tracking-wider font-semibold mb-1">🏆 Acumulado</p>
                <p class="text-4xl md:text-6xl font-black text-yellow-400">
                    $<span x-text="Number(acumulado).toLocaleString()">{{ number_format($juego->acumulado ?? 0, 0) }}</span>
                </p>
            </div>
            <div class="glass-effect rounded-2xl p-4 md:p-6 text-center border-2 border-purple-500/30 bg-gradient-to-br from-purple-500/10 to-pink-500/10">
                <p class="text-sm md:text-base text-purple-400 uppercase tracking-wider font-semibold mb-1">🎰 Bingo</p>
                <p class="text-3xl md:text-5xl font-black text-purple-400">
                    $<span x-text="Number(premioBingo).toLocaleString()">{{ number_format($juego->premio_bingo ?? 0, 0) }}</span>
                </p>
            </div>
        </div>

        <div class="glass-effect rounded-2xl p-5 md:p-8 relative overflow-hidden">
            <div class="absolute inset-0 bg-gradient-to-br from-blue-600/20 via-purple-600/20 to-pink-600/20"></div>
            
            <!-- Indicador EN VIVO -->
            <div x-show="sorteoEnVivo" 
                 x-transition
                 class="absolute top-3 right-3 flex items-center gap-2 px-3 py-1.5 rounded-full bg-red-500/90 text-white text-xs font-bold uppercase tracking-wider animate-pulse z-10">
                <span class="w-2 h-2 rounded-full bg-white animate-ping"></span>
                <span>EN VIVO</span>
            </div>
            
            <!-- Indicador de controlador del sorteo auto -->
            <div x-show="sorteoAutoActivo && !esControlador" 
                 x-transition
                 class="absolute top-3 left-3 flex items-center gap-2 px-3 py-1.5 rounded-full bg-indigo-500/90 text-white text-xs font-semibold z-10">
                <span class="w-2 h-2 rounded-full bg-yellow-400 animate-pulse"></span>
                <span x-text="controladorNombre + ' controla'"></span>
            </div>
            
            <div class="relative">
                <div class="flex flex-col lg:flex-row items-center gap-6 lg:gap-10">
                    
                    <div class="flex items-center gap-8">
                        <div class="relative">
                            <div id="bombo" class="w-36 h-36 md:w-44 md:h-44 lg:w-48 lg:h-48 rounded-full bg-gradient-to-br from-gray-800 to-gray-900 border-4 border-yellow-500/50 flex items-center justify-center relative overflow-hidden shadow-2xl shadow-yellow-500/20"
                                 :class="{ 'animate-shake': isDrawing }">
                                <div class="absolute inset-4 rounded-full border-2 border-dashed border-yellow-500/30 animate-spin" style="animation-duration: 8s;"></div>
                                
                                <div x-show="!ultimaBola && !isDrawing" class="text-center">
                                    <p class="text-6xl md:text-7xl font-black text-yellow-500/50">?</p>
                                </div>
                                
                                <div x-show="isDrawing" class="text-center relative">
                                    <div class="w-20 h-20 rounded-full bg-gradient-to-br from-yellow-300 via-white to-yellow-400 animate-spin shadow-2xl shadow-yellow-500/50"
                                         style="animation-duration: 0.3s;">
                                        <div class="absolute inset-2 rounded-full bg-gradient-to-br from-white to-gray-100"></div>
                                    </div>
                                    <div class="absolute inset-0 rounded-full border-4 border-yellow-400/50 animate-ping"></div>
                                </div>
                                
                                <div x-show="ultimaBola && !isDrawing" class="text-center relative">
                                    <div class="absolute inset-0 flex items-center justify-center">
                                        <div class="w-40 h-40 rounded-full border-2 border-yellow-400/30 animate-ring-expand"></div>
                                    </div>
                                    <div class="absolute inset-0 flex items-center justify-center" style="animation-delay: 0.2s;">
                                        <div class="w-40 h-40 rounded-full border-2 border-yellow-400/20 animate-ring-expand" style="animation-delay: 0.2s;"></div>
                                    </div>
                                    
                                    <div class="w-28 h-28 md:w-32 md:h-32 lg:w-36 lg:h-36 rounded-full flex items-center justify-center bingo-ball-3d animate-ball-bounce-in relative"
                                         :class="{
                                             'bg-gradient-to-br from-blue-400 via-blue-500 to-blue-700 text-blue-400': ultimaBola?.letra === 'B',
                                             'bg-gradient-to-br from-red-400 via-red-500 to-red-700 text-red-400': ultimaBola?.letra === 'I',
                                             'bg-gradient-to-br from-gray-100 via-white to-gray-200 text-gray-400': ultimaBola?.letra === 'N',
                                             'bg-gradient-to-br from-green-400 via-green-500 to-green-700 text-green-400': ultimaBola?.letra === 'G',
                                             'bg-gradient-to-br from-yellow-300 via-yellow-400 to-orange-500 text-yellow-400': ultimaBola?.letra === 'O'
                                         }">
                                        <div class="absolute inset-0 rounded-full animate-ball-shine opacity-50"></div>
                                        
                                        <div class="w-20 h-20 md:w-24 md:h-24 rounded-full bg-white flex items-center justify-center shadow-inner relative z-10">
                                            <div class="text-center">
                                                <p class="text-sm md:text-base font-bold text-gray-500" x-text="ultimaBola?.letra"></p>
                                                <p class="text-4xl md:text-5xl font-black text-gray-800" x-text="ultimaBola?.numero"></p>
                                            </div>
                                        </div>
                                        
                                        <div class="absolute -top-1 -right-1 w-4 h-4 bg-white/60 rounded-full blur-sm"></div>
                                    </div>
                                </div>
                            </div>
                            <div class="absolute -bottom-2 left-1/2 -translate-x-1/2 w-24 h-6 bg-gradient-to-t from-yellow-600/50 to-yellow-500/30 rounded-full blur-md"></div>
                        </div>

                        <div class="text-center lg:text-left">
                            <p class="text-sm text-gray-400 uppercase tracking-wider">Última Bola</p>
                            <p class="text-5xl md:text-6xl font-black text-white" x-text="ultimaBola ? ultimaBola.display : '---'">---</p>
                            <p class="text-base text-gray-400 mt-2">Bola <span class="text-yellow-400 font-bold text-xl" x-text="bolasSorteadas"></span> de <span x-text="maxBolas">60</span></p>
                            
                            <div class="flex flex-wrap items-center justify-center lg:justify-start gap-2 mt-4">
                                <!-- Indicador de estado del sorteo para jugadores -->
                                <div x-show="sorteoEnVivo" class="px-4 py-2 rounded-xl bg-gradient-to-r from-green-500/20 to-emerald-500/20 border border-green-500/50">
                                    <span class="text-green-400 font-semibold text-sm flex items-center gap-2">
                                        <span class="w-2 h-2 rounded-full bg-green-500 animate-pulse"></span>
                                        Sorteo en curso
                                    </span>
                                </div>
                                
                                <div x-show="!sorteoEnVivo && estado === 'esperando'" class="px-4 py-2 rounded-xl bg-gradient-to-r from-yellow-500/20 to-orange-500/20 border border-yellow-500/50">
                                    <span class="text-yellow-400 font-semibold text-sm">⏳ Esperando inicio...</span>
                                </div>

                                <button @click="showExitModal = true"
                                        title="Abandonar el juego y volver al inicio"
                                        aria-label="Salir del juego"
                                        class="px-4 sm:px-6 py-2 sm:py-3 rounded-xl text-sm sm:text-base font-semibold text-white bg-gray-600 hover:bg-gray-500 transition-colors flex items-center gap-2">
                                    🚪 <span>Salir</span>
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="flex-1 w-full lg:w-auto">
                        <div class="flex items-center justify-between mb-3">
                            <p class="text-sm text-gray-400 uppercase tracking-wider">Últimas Bolas</p>
                            <span class="text-xs text-yellow-400 font-bold" x-text="bolasSorteadas + '/' + maxBolas"></span>
                        </div>
                        
                        <div class="flex flex-wrap gap-2 justify-center lg:justify-start mb-4">
                            <template x-for="(bola, index) in historial.slice(-10).reverse()" :key="bola.orden">
                                <div class="relative transition-all duration-300" 
                                     :class="index === 0 ? 'scale-110 z-10' : ''">
                                    <div class="w-11 h-11 md:w-12 md:h-12 rounded-full flex items-center justify-center bingo-ball-3d relative"
                                         :class="{
                                             'bg-gradient-to-br from-blue-400 to-blue-600 shadow-blue-500/50': bola.letra === 'B',
                                             'bg-gradient-to-br from-red-400 to-red-600 shadow-red-500/50': bola.letra === 'I',
                                             'bg-gradient-to-br from-gray-200 to-gray-300 shadow-gray-400/50': bola.letra === 'N',
                                             'bg-gradient-to-br from-green-400 to-green-600 shadow-green-500/50': bola.letra === 'G',
                                             'bg-gradient-to-br from-yellow-300 to-orange-500 shadow-yellow-500/50': bola.letra === 'O',
                                             'animate-ball-emerge shadow-lg': index === 0
                                         }">
                                        <div class="w-7 h-7 md:w-8 md:h-8 rounded-full bg-white flex items-center justify-center shadow-inner">
                                            <span class="text-sm md:text-base font-black text-gray-800" x-text="bola.numero"></span>
                                        </div>
                                    </div>
                                    <template x-if="index === 0">
                                        <div class="absolute -top-1 -right-1 w-4 h-4 bg-yellow-400 rounded-full flex items-center justify-center text-[8px] animate-bounce">★</div>
                                    </template>
                                </div>
                            </template>
                            <template x-if="historial.length === 0">
                                <p class="text-gray-500 text-sm">Esperando sorteo...</p>
                            </template>
                        </div>

                        <div class="glass-effect rounded-lg p-1.5 sm:p-2 bg-gray-900/50 overflow-x-auto">
                            <div class="space-y-0.5 sm:space-y-1 min-w-[300px]">
                                <template x-for="letra in ['B', 'I', 'N', 'G', 'O']" :key="letra">
                                    <div class="flex items-center gap-0.5 sm:gap-1">
                                        <div class="w-5 h-5 sm:w-7 sm:h-7 rounded flex items-center justify-center text-[10px] sm:text-xs font-bold shrink-0"
                                             :class="{
                                                 'bg-blue-500 text-white': letra === 'B',
                                                 'bg-red-500 text-white': letra === 'I',
                                                 'bg-gray-400 text-gray-800': letra === 'N',
                                                 'bg-green-500 text-white': letra === 'G',
                                                 'bg-yellow-500 text-gray-800': letra === 'O'
                                             }" x-text="letra"></div>
                                        <div class="grid grid-cols-15 gap-px sm:gap-0.5 flex-1">
                                            <template x-for="num in getNumerosPorLetra(letra)" :key="num">
                                                <div class="aspect-square rounded text-[7px] sm:text-[10px] md:text-xs flex items-center justify-center transition-all font-semibold"
                                                     :class="numerosSorteados.includes(num) 
                                                         ? (letra === 'B' ? 'bg-blue-500 text-white font-bold' :
                                                            letra === 'I' ? 'bg-red-500 text-white font-bold' :
                                                            letra === 'N' ? 'bg-gray-200 text-gray-800 font-bold' :
                                                            letra === 'G' ? 'bg-green-500 text-white font-bold' :
                                                            'bg-yellow-400 text-gray-800 font-bold')
                                                         : 'bg-gray-800/80 text-gray-500'"
                                                     x-text="num"></div>
                                            </template>
                                        </div>
                                    </div>
                                </template>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="glass-effect rounded-xl p-2 sm:p-4">
            <div class="flex items-center justify-between mb-2 sm:mb-3">
                <h3 class="text-xs sm:text-sm font-semibold text-cyan-400 uppercase tracking-wider">Tus Cartones</h3>
                <span class="text-[10px] sm:text-xs text-gray-400" x-text="cartones.length + ' / 4'"></span>
            </div>
            
            <div class="grid grid-cols-2 lg:grid-cols-4 gap-2 sm:gap-3">
                <template x-for="carton in cartones" :key="carton.id">
                    <div class="rounded-xl overflow-hidden transition-all duration-300"
                         :class="{
                             'ring-4 ring-yellow-400 shadow-2xl shadow-yellow-500/50 scale-105 z-10': carton.tiene_bingo,
                             'ring-4 ring-green-400 shadow-xl shadow-green-500/40': carton.tiene_linea && !carton.tiene_bingo,
                             'border-2 border-gray-700': !carton.tiene_linea && !carton.tiene_bingo
                         }">
                        
                        <div class="px-2 py-1.5 relative"
                             :class="{
                                 'bg-gradient-to-r from-yellow-500 via-orange-500 to-yellow-500 animate-gradient-x': carton.tiene_bingo,
                                 'bg-gradient-to-r from-green-600 to-green-500': carton.tiene_linea && !carton.tiene_bingo,
                                 'bg-gradient-to-r from-gray-800 to-gray-900': !carton.tiene_linea && !carton.tiene_bingo
                             }">
                            <!-- Número del cartón (izquierda) -->
                            <span class="absolute left-2 top-1/2 -translate-y-1/2 text-[10px] font-semibold" 
                                  :class="carton.tiene_bingo || carton.tiene_linea ? 'text-white' : 'text-gray-400'">
                                #<span x-text="carton.posicion"></span>
                            </span>
                            
                            <!-- Badge centrado cuando hay línea o bingo -->
                            <div class="flex justify-center">
                                <template x-if="carton.tiene_linea && !carton.tiene_bingo">
                                    <div class="flex items-center gap-1 px-2 py-0.5 rounded bg-white shadow-md">
                                        <svg class="w-3.5 h-3.5 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="3" d="M5 13l4 4L19 7"/>
                                        </svg>
                                        <span class="text-[9px] font-black text-green-600">¡LÍNEA!</span>
                                    </div>
                                </template>
                                
                                <template x-if="carton.tiene_bingo">
                                    <div class="flex items-center gap-1 px-2 py-0.5 rounded bg-white shadow-md animate-pulse">
                                        <span class="text-base">🏆</span>
                                        <span class="text-[9px] font-black text-yellow-600">¡BINGO!</span>
                                    </div>
                                </template>
                                
                                <template x-if="!carton.tiene_linea && !carton.tiene_bingo">
                                    <span class="text-[10px] text-gray-500">Cartón</span>
                                </template>
                            </div>
                            
                            <!-- Apuesta (derecha) -->
                            <span class="absolute right-2 top-1/2 -translate-y-1/2 text-[9px] font-bold px-1.5 py-0.5 rounded bg-black/20"
                                  :class="carton.tiene_bingo || carton.tiene_linea ? 'text-white' : 'text-yellow-400'">
                                x<span x-text="carton.apuesta"></span>
                            </span>
                        </div>
                        
                        <div class="p-1" :class="carton.tiene_bingo ? 'bg-yellow-50' : (carton.tiene_linea ? 'bg-green-50' : 'bg-white')">
                            <div class="grid grid-cols-5 gap-px">
                                <template x-for="letra in ['B', 'I', 'N', 'G', 'O']" :key="letra">
                                    <div class="text-center py-0.5 text-[10px] font-bold"
                                         :class="{
                                             'bg-blue-500 text-white': letra === 'B',
                                             'bg-red-500 text-white': letra === 'I',
                                             'bg-gray-300 text-gray-800': letra === 'N',
                                             'bg-green-500 text-white': letra === 'G',
                                             'bg-yellow-500 text-gray-800': letra === 'O'
                                         }" x-text="letra"></div>
                                </template>
                            </div>
                            
                            <template x-for="fila in [0, 1, 2, 3, 4]" :key="fila">
                                <div class="grid grid-cols-5 gap-px">
                                    <template x-for="letra in ['B', 'I', 'N', 'G', 'O']" :key="letra">
                                        <div class="aspect-square flex items-center justify-center text-sm font-bold carton-cell"
                                             :class="getCartonCellClass(carton, letra, fila)"
                                             x-text="carton.numeros[letra][fila] === 0 ? '★' : carton.numeros[letra][fila]">
                                        </div>
                                    </template>
                                </div>
                            </template>
                        </div>

                        <div class="p-2 border-t transition-all duration-300"
                             :class="estado === 'esperando' 
                                 ? 'bg-gradient-to-r from-orange-500/20 to-yellow-500/20 border-orange-500/30' 
                                 : 'bg-gray-800/50 border-gray-700/50'">
                            <div class="flex items-center justify-between gap-2">
                                <button @click="ajustarApuesta(carton.id, -1)"
                                        :disabled="estado !== 'esperando' || ajustandoCarton === carton.id || carton.apuesta <= 1"
                                        title="Reducir apuesta en 1x"
                                        aria-label="Reducir apuesta"
                                        class="w-8 h-8 rounded-lg font-bold text-lg transition-all disabled:opacity-30 disabled:cursor-not-allowed"
                                        :class="estado === 'esperando' 
                                            ? 'bg-red-500 text-white shadow shadow-red-500/30 hover:scale-110 active:scale-95' 
                                            : 'bg-gray-600 text-gray-400'">
                                    −
                                </button>
                                
                                <div class="flex-1 text-center" title="Multiplicador de tu apuesta base">
                                    <p class="text-[9px] uppercase transition-colors"
                                       :class="estado === 'esperando' ? 'text-gray-400' : 'text-gray-500'">Apuesta</p>
                                    <p class="text-lg font-black transition-colors"
                                       :class="estado === 'esperando' ? 'text-yellow-400' : 'text-gray-400'">
                                        x<span x-text="carton.apuesta"></span>
                                    </p>
                                </div>
                                
                                <button @click="ajustarApuesta(carton.id, 1)"
                                        :disabled="estado !== 'esperando' || ajustandoCarton === carton.id"
                                        title="Aumentar apuesta en 1x"
                                        aria-label="Aumentar apuesta"
                                        class="w-8 h-8 rounded-lg font-bold text-lg transition-all disabled:opacity-30 disabled:cursor-not-allowed"
                                        :class="estado === 'esperando' 
                                            ? 'bg-green-500 text-white shadow shadow-green-500/30 hover:scale-110 active:scale-95' 
                                            : 'bg-gray-600 text-gray-400'">
                                    +
                                </button>
                            </div>
                            <div class="flex flex-wrap justify-center gap-1.5 mt-2">
                                <template x-for="mult in [10, 20, 30, 40, 50]" :key="mult">
                                    <button @click="multiplicarCarton(carton.id, mult)"
                                            :disabled="estado !== 'esperando' || multiplicandoCarton === carton.id"
                                            :title="'Establecer apuesta a ' + mult + 'x'"
                                            :aria-label="'Multiplicar por ' + mult"
                                            class="px-2.5 py-1 rounded text-[10px] font-bold transition-all disabled:opacity-30 disabled:cursor-not-allowed"
                                            :class="estado === 'esperando' 
                                                ? 'bg-gray-700 text-yellow-400 hover:bg-gray-600 hover:scale-110 active:scale-95' 
                                                : 'bg-gray-700/50 text-gray-500'">
                                        x<span x-text="mult"></span>
                                    </button>
                                </template>
                            </div>
                            <p x-show="estado !== 'esperando'" 
                               class="text-[8px] text-center text-gray-500 mt-1 uppercase">
                                🔒 Bloqueado
                            </p>
                        </div>
                    </div>
                </template>
                
                <template x-if="cartones.length === 0">
                    <div class="col-span-4 text-center py-12">
                        <p class="text-gray-500 text-sm mb-2">No tienes cartones</p>
                        <a href="{{ route('bingo.index') }}" class="inline-block px-6 py-2 rounded-lg bg-yellow-500 text-gray-900 font-semibold text-sm hover:bg-yellow-400 transition-colors">
                            Comprar cartones
                        </a>
                    </div>
                </template>
            </div>
        </div>
    </div>

    <!-- Contenedor de Confeti -->
    <div x-ref="confettiContainer" class="fixed inset-0 pointer-events-none z-[100] overflow-hidden"></div>

    <div x-cloak
         x-show="showWinModal" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80"
         @click.self="showWinModal = false; stopConfetti()">
        <div class="glass-effect rounded-3xl p-8 max-w-md w-full text-center animate-bounce-ball relative">
            <div class="mb-6">
                <div class="w-24 h-24 mx-auto rounded-full flex items-center justify-center shadow-2xl animate-pulse-glow"
                     :class="winType === 'bingo' ? 'bg-gradient-to-br from-yellow-400 to-orange-500 shadow-yellow-500/50' : 'bg-gradient-to-br from-green-400 to-emerald-500 shadow-green-500/50'">
                    <span class="text-5xl" x-text="winType === 'bingo' ? '🏆' : '🎯'"></span>
                </div>
            </div>
            <h2 class="text-4xl font-black mb-2" 
                :class="winType === 'bingo' ? 'text-yellow-400' : 'text-green-400'"
                x-text="winType === 'bingo' ? '¡¡¡BINGO!!!' : '¡LÍNEA!'"></h2>
            <p class="text-gray-400 mb-4" x-text="winType === 'bingo' ? '¡Felicitaciones! ¡Has ganado el BINGO!' : '¡Felicitaciones! ¡Has completado una línea!'"></p>
            <p class="text-5xl font-black text-green-400 mb-6">$<span x-text="winAmount.toLocaleString()"></span></p>
            <button @click="showWinModal = false; stopConfetti()" 
                    class="px-8 py-3 rounded-xl font-bold text-gray-900 hover:scale-105 transition-all"
                    :class="winType === 'bingo' ? 'bg-gradient-to-r from-yellow-500 to-orange-500 hover:from-yellow-400 hover:to-orange-400' : 'bg-gradient-to-r from-green-500 to-emerald-500 hover:from-green-400 hover:to-emerald-400'">
                ¡Genial!
            </button>
        </div>
    </div>

    <div x-cloak
         x-show="showExitModal" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80"
         @click.self="showExitModal = false">
        <div class="glass-effect rounded-3xl p-8 max-w-md w-full text-center">
            <div class="mb-6">
                <div class="w-20 h-20 mx-auto rounded-full bg-gradient-to-br from-red-500 to-orange-500 flex items-center justify-center shadow-2xl">
                    <span class="text-4xl">🚪</span>
                </div>
            </div>
            <h2 class="text-2xl font-black text-white mb-2">¿Salir del juego?</h2>
            <p class="text-gray-400 mb-6">Si abandonas, perderás todos tus cartones y no podrás recuperarlos.</p>
            
            <div class="flex gap-4 justify-center">
                <button @click="showExitModal = false" 
                        class="px-6 py-3 rounded-xl font-bold bg-gray-600 text-white hover:bg-gray-500 transition-all">
                    Cancelar
                </button>
                <form action="{{ route('bingo.abandonar', $juego) }}" method="POST" class="inline">
                    @csrf
                    <button type="submit" 
                            class="px-6 py-3 rounded-xl font-bold bg-gradient-to-r from-red-500 to-orange-500 text-white hover:from-red-400 hover:to-orange-400 transition-all">
                        Sí, abandonar
                    </button>
                </form>
            </div>
        </div>
    </div>


    <!-- Modal de juego finalizado -->
    <div x-cloak
         x-show="showGameOverModal" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         class="fixed inset-0 z-50 flex items-center justify-center p-4 bg-black/80">
        <div class="glass-effect rounded-3xl p-8 max-w-md w-full text-center animate-slide-up">
            <div class="mb-6">
                <div class="w-24 h-24 mx-auto rounded-full bg-gradient-to-br from-gray-600 to-gray-800 flex items-center justify-center shadow-2xl border-4 border-gray-500">
                    <span class="text-5xl">🎰</span>
                </div>
            </div>
            <h2 class="text-3xl font-black text-white mb-3">¡Sorteo Finalizado!</h2>
            <p class="text-gray-400 mb-2">Se han sorteado las <span class="text-yellow-400 font-bold" x-text="maxBolas">60</span> bolas del juego.</p>
            <p class="text-gray-500 text-sm mb-6">Ningún jugador logró completar el cartón esta vez.</p>
            
            <div class="glass-effect rounded-xl p-4 mb-6 bg-gradient-to-r from-blue-500/10 to-purple-500/10 border border-blue-500/30">
                <p class="text-sm text-gray-400 mb-1">Bolas sorteadas</p>
                <p class="text-3xl font-black text-blue-400"><span x-text="bolasSorteadas">60</span> / <span x-text="maxBolas">60</span></p>
            </div>

            <div class="flex flex-col gap-3">
                <a href="{{ route('bingo.index') }}" 
                   class="w-full px-6 py-3 rounded-xl font-bold bg-gradient-to-r from-yellow-500 to-orange-500 text-gray-900 hover:from-yellow-400 hover:to-orange-400 transition-all text-center">
                    🎫 Nuevo Juego
                </a>
                <button @click="showGameOverModal = false" 
                        class="w-full px-6 py-3 rounded-xl font-bold bg-gray-700 text-white hover:bg-gray-600 transition-all">
                    Ver Resultados
                </button>
            </div>
        </div>
    </div>

    <!-- CONTEO REGRESIVO ÉPICO -->
    <div x-cloak
         x-show="showCountdown" 
         x-transition:enter="transition ease-out duration-300"
         x-transition:enter-start="opacity-0"
         x-transition:enter-end="opacity-100"
         x-transition:leave="transition ease-in duration-300"
         x-transition:leave-start="opacity-100"
         x-transition:leave-end="opacity-0"
         class="fixed inset-0 z-[100] flex items-center justify-center"
         style="background: radial-gradient(circle at center, rgba(0,0,0,0.9) 0%, rgba(0,0,0,0.98) 100%);">
        
        <!-- Efectos de fondo -->
        <div class="absolute inset-0 overflow-hidden pointer-events-none">
            <div class="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[600px] h-[600px] bg-yellow-500/20 rounded-full blur-3xl animate-pulse"></div>
            <div class="absolute top-20 left-20 text-6xl animate-bounce" style="animation-delay: 0.1s;">🎱</div>
            <div class="absolute top-32 right-24 text-5xl animate-bounce" style="animation-delay: 0.3s;">🎰</div>
            <div class="absolute bottom-28 left-32 text-5xl animate-bounce" style="animation-delay: 0.5s;">🍀</div>
            <div class="absolute bottom-20 right-20 text-6xl animate-bounce" style="animation-delay: 0.2s;">💰</div>
        </div>

        <div class="relative text-center px-4">
            <!-- Badge de apuestas cerradas -->
            <div class="mb-6">
                <div class="inline-flex items-center gap-2 px-5 py-2 rounded-full bg-red-600 border border-red-400 shadow-lg shadow-red-500/40 animate-pulse">
                    <span class="text-xl">🔒</span>
                    <span class="text-white font-bold text-sm uppercase tracking-wider">Apuestas Cerradas</span>
                </div>
            </div>

            <!-- Emoji grande animado -->
            <div class="text-8xl md:text-9xl mb-4 animate-bounce" x-text="countdownEmoji"></div>

            <!-- Frase principal -->
            <div class="relative mb-6">
                <h1 class="text-4xl md:text-6xl font-black text-transparent bg-clip-text bg-gradient-to-r from-yellow-400 via-orange-500 to-red-500 animate-pulse"
                    x-text="countdownPhrase"
                    style="text-shadow: 0 0 40px rgba(251, 146, 60, 0.5);"></h1>
            </div>

            <!-- Barras de progreso animadas -->
            <div class="flex justify-center gap-3 mb-6">
                <div class="w-16 h-2 rounded-full bg-gray-700 overflow-hidden">
                    <div class="h-full bg-gradient-to-r from-yellow-400 to-yellow-500 animate-pulse" 
                         :class="countdownPhrase.includes('PREPARADOS') ? 'w-full' : 'w-0'"
                         style="transition: width 0.3s;"></div>
                </div>
                <div class="w-16 h-2 rounded-full bg-gray-700 overflow-hidden">
                    <div class="h-full bg-gradient-to-r from-orange-400 to-orange-500 animate-pulse" 
                         :class="countdownPhrase.includes('LISTOS') ? 'w-full' : 'w-0'"
                         style="transition: width 0.3s;"></div>
                </div>
                <div class="w-16 h-2 rounded-full bg-gray-700 overflow-hidden">
                    <div class="h-full bg-gradient-to-r from-red-400 to-red-500 animate-pulse" 
                         :class="countdownPhrase.includes('EMPEZAMOS') ? 'w-full' : 'w-0'"
                         style="transition: width 0.3s;"></div>
                </div>
            </div>

            <!-- Mensaje -->
            <p class="text-lg text-gray-300 mb-2">Las bolas comenzarán a salir...</p>
            <p class="text-yellow-400/80 text-sm">🍀 ¡Buena suerte! 🍀</p>
        </div>
    </div>
</div>

@push('scripts')
<script>
function bingoGame() {
    return {
        juegoId: {{ $juego->id }},
        estado: '{{ $juego->estado }}',
        acumulado: {{ (float) ($juego->acumulado ?? 0) }},
        premioLinea: {{ (float) ($juego->premio_linea ?? 0) }},
        premioBingo: {{ (float) ($juego->premio_bingo ?? 0) }},
        bolasSorteadas: {{ $bolasSorteadas->count() }},
        maxBolas: {{ $juego->getMaxBolas() }},
        ultimaBola: @json($ultimaBolaJson),
        cartones: @json($cartonesJson),
        historial: @json($historialJson),
        numerosSorteados: @json($numerosSorteados),
        isDrawing: false,
        autoPlayInterval: null,
        showWinModal: false,
        showExitModal: false,
        showGameOverModal: false,
        winType: '',
        winAmount: 0,
        multiplicandoCarton: null,
        ajustandoCarton: null,
        mensajeMultiplicador: '',
        numeroRecienSorteado: null,
        balance: {{ (float) $balance }},
        ultimaGanancia: 0,
        nombreJugador: '{{ $nombreJugador }}',
        sessionId: '{{ Session::getId() }}',
        jugadores: @json($jugadoresActivos),
        ganadorNombre: '',
        sorteoEnVivo: {{ $juego->estado === 'en_curso' ? 'true' : 'false' }},
        sorteoAutoActivo: false,
        esControlador: false,
        controladorNombre: '',
        showCountdown: false,
        countdownPhrase: '',
        countdownEmoji: '🎰',

        init() {
            if (this.estado === 'finalizado') {
                window.location.href = '{{ route("bingo.index") }}';
                return;
            }

            this.setupWebSocket();
            this.verificarSorteoAutoActivo();
        },

        async verificarSorteoAutoActivo() {
            try {
                const response = await fetch(`/sorteo-auto/estado/${this.juegoId}`);
                const data = await response.json();
                
                if (data.activo) {
                    this.sorteoAutoActivo = true;
                    this.esControlador = data.es_controlador;
                    this.controladorNombre = data.controlador;
                    this.sorteoEnVivo = true;
                    
                    if (this.esControlador) {
                        this.startAutoPlay();
                    }
                }
            } catch (error) {
                console.error('Error verificando sorteo auto:', error);
            }
        },

        setupWebSocket() {
            if (typeof window.Echo === 'undefined') {
                console.warn('Echo no disponible, modo offline');
                this.mostrarNotificacion('⚠️ WebSocket no conectado - modo offline');
                return;
            }

            console.log('🔌 Conectando a WebSocket canal: juego.' + this.juegoId);

            const channel = window.Echo.channel(`juego.${this.juegoId}`);
            
            channel
                .listen('.bola.sorteada', (e) => {
                    console.log('📡 Evento bola.sorteada recibido');
                    this.handleBolaSorteada(e);
                })
                .listen('.juego.actualizado', (e) => {
                    console.log('📡 Evento juego.actualizado recibido');
                    this.handleJuegoActualizado(e);
                })
                .listen('.sorteo.iniciado', (e) => {
                    console.log('📡 Evento sorteo.iniciado recibido');
                    this.handleSorteoIniciado(e);
                })
                .listen('.sorteo.auto.iniciado', (e) => {
                    console.log('📡 Evento sorteo.auto.iniciado recibido');
                    this.handleSorteoAutoIniciado(e);
                })
                .listen('.sorteo.auto.detenido', (e) => {
                    console.log('📡 Evento sorteo.auto.detenido recibido');
                    this.handleSorteoAutoDetenido(e);
                })
                .listen('.mensaje.admin', (e) => {
                    console.log('📡 Evento mensaje.admin recibido');
                    this.handleMensajeAdmin(e);
                })
                .listen('.juego.reiniciado', (e) => {
                    console.log('📡 Evento juego.reiniciado recibido');
                    this.handleJuegoReiniciado(e);
                })
                .listen('.configuracion.actualizada', (e) => {
                    console.log('📡 Evento configuracion.actualizada recibido');
                    this.handleConfiguracionActualizada(e);
                });
            
            console.log('✅ Escuchando eventos en canal juego.' + this.juegoId);
        },

        handleConfiguracionActualizada(data) {
            console.log('⚙️ Configuración actualizada:', data);
            this.maxBolas = data.maxBolas;
            if (data.mensaje) {
                this.mostrarNotificacion(`⚙️ ${data.mensaje}`, 'info');
            }
        },

        handleSorteoAutoIniciado(data) {
            console.log('🎲 Sorteo auto iniciado:', data);
            this.sorteoAutoActivo = true;
            this.controladorNombre = data.controladorNombre;
            this.esControlador = data.controladorId === this.sessionId;
            
            // Iniciar conteo regresivo épico para TODOS los jugadores
            this.iniciarConteoRegresivo(() => {
                this.estado = 'en_curso';
                this.sorteoEnVivo = true;
            });
        },

        iniciarConteoRegresivo(callback) {
            this.showCountdown = true;
            const frases = [
                { texto: '¡PREPARADOS!', emoji: '🎯' },
                { texto: '¡LISTOS!', emoji: '🔥' },
                { texto: '¡EMPEZAMOS!', emoji: '🚀' }
            ];
            let index = 0;
            this.countdownPhrase = frases[index].texto;
            this.countdownEmoji = frases[index].emoji;
            
            const countInterval = setInterval(() => {
                index++;
                if (index < frases.length) {
                    this.countdownPhrase = frases[index].texto;
                    this.countdownEmoji = frases[index].emoji;
                } else {
                    clearInterval(countInterval);
                    this.showCountdown = false;
                    if (callback) callback();
                }
            }, 1000);
        },

        handleSorteoAutoDetenido(data) {
            console.log('🛑 Sorteo detenido recibido:', data);
            this.sorteoAutoActivo = false;
            this.esControlador = false;
            this.controladorNombre = '';
            this.sorteoEnVivo = false;
            this.stopAutoPlay();
            
            this.mostrarNotificacion('⏹️ Sorteo automático detenido');
        },

        handleJuegoReiniciado(data) {
            console.log('🔄 Juego reiniciado:', data);
            this.mostrarMensajeAdmin(data.mensaje, 'alerta');
            
            setTimeout(() => {
                window.location.href = '{{ route("bingo.index") }}';
            }, 3000);
        },

        handleMensajeAdmin(data) {
            console.log('📢 Mensaje del admin recibido:', data);
            this.mostrarMensajeAdmin(data.mensaje, data.tipo, data.gif);
        },

        mostrarMensajeAdmin(mensaje, tipo = 'info', gif = null) {
            const colores = {
                info: 'from-blue-500/90 to-cyan-500/90 border-blue-400',
                alerta: 'from-yellow-500/90 to-orange-500/90 border-yellow-400',
                exito: 'from-green-500/90 to-emerald-500/90 border-green-400',
                promo: 'from-purple-500/90 to-pink-500/90 border-purple-400',
            };
            
            const iconos = {
                info: 'ℹ️',
                alerta: '⚠️',
                exito: '✅',
                promo: '🎁',
            };

            const gifHtml = gif ? `
                <div class="mt-3 rounded-xl overflow-hidden border-2 border-white/30 shadow-lg">
                    <img src="${gif}" alt="GIF" class="w-full max-h-48 object-cover" loading="lazy">
                </div>
            ` : '';

            const container = document.createElement('div');
            container.className = `fixed top-1/4 left-1/2 -translate-x-1/2 z-[200] max-w-md w-full mx-4`;
            container.innerHTML = `
                <div class="bg-gradient-to-br ${colores[tipo]} backdrop-blur-xl rounded-2xl p-5 border-2 shadow-2xl animate-slide-up" style="animation: slideUp 0.5s ease-out;">
                    <div class="flex items-start gap-3">
                        <span class="text-4xl animate-bounce">${iconos[tipo]}</span>
                        <div class="flex-1">
                            <p class="text-xs font-bold text-white/80 uppercase tracking-wider mb-1">🎰 Mensaje del Moderador</p>
                            <p class="text-white font-bold text-xl leading-tight">${mensaje}</p>
                        </div>
                    </div>
                    ${gifHtml}
                    <div class="absolute -top-2 -right-2 w-6 h-6 rounded-full bg-white/30 animate-ping"></div>
                </div>
            `;
            document.body.appendChild(container);
            
            const duracion = gif ? 7000 : 5000;
            
            setTimeout(() => {
                container.style.transition = 'all 0.5s ease-out';
                container.style.opacity = '0';
                container.style.transform = 'translate(-50%, -30px) scale(0.9)';
                setTimeout(() => container.remove(), 500);
            }, duracion);
        },

        handleSorteoIniciado(data) {
            this.estado = data.estado;
            this.sorteoEnVivo = true;
            this.mostrarNotificacion(`🎰 ${data.iniciadoPor} inició el sorteo!`);
        },

        handleBolaSorteada(data) {
            console.log('🎱 Bola recibida via WebSocket:', data.bola);
            
            this.isDrawing = true;
            this.sorteoEnVivo = true;
            
            setTimeout(() => {
                this.ultimaBola = data.bola;
                this.bolasSorteadas = data.bolasSorteadas;
                if (!this.numerosSorteados.includes(data.bola.numero)) {
                    this.numerosSorteados.push(data.bola.numero);
                    this.historial.push(data.bola);
                }
                this.acumulado = data.acumulado;
                this.premioLinea = data.premioLinea;
                this.premioBingo = data.premioBingo;
                
                if (data.estado === 'en_curso' && this.estado === 'esperando') {
                    this.mostrarNotificacion('🔒 ¡Apuestas cerradas!', 'alerta');
                }
                this.estado = data.estado;

                this.numeroRecienSorteado = data.bola.numero;
                
                setTimeout(() => {
                    this.marcarEnCartones(data.bola.numero);
                    this.numeroRecienSorteado = null;
                }, 1500);
                
                this.isDrawing = false;

                if (data.ganadorLinea) {
                    this.mostrarGanador('linea', data.ganadorLinea);
                }

                if (data.ganadorBingo) {
                    this.mostrarGanador('bingo', data.ganadorBingo);
                    this.estado = 'finalizado';
                    this.sorteoAutoActivo = false;
                    this.stopAutoPlay();
                }

                if (data.finalizado && !data.ganadorBingo) {
                    this.estado = 'finalizado';
                    this.sorteoAutoActivo = false;
                    this.stopAutoPlay();
                    if (data.limiteAlcanzado) {
                        setTimeout(() => {
                            this.showGameOverModal = true;
                        }, 500);
                    }
                }
            }, 800);
        },

        handleJuegoActualizado(data) {
            this.acumulado = data.acumulado;
            this.premioLinea = data.premioLinea;
            this.premioBingo = data.premioBingo;
            this.estado = data.estado;
            
            if (data.jugadores) {
                this.jugadores = data.jugadores.filter(j => j.nombre !== this.nombreJugador);
            }
            
            if (data.nuevoJugador && data.tipo === 'carton_comprado') {
                this.mostrarNotificacion(`🎮 ${data.nuevoJugador} se unió al juego!`);
            }
        },

        mostrarNotificacion(mensaje) {
            const notif = document.createElement('div');
            notif.className = 'fixed top-20 left-1/2 -translate-x-1/2 z-50 px-4 py-2 rounded-xl bg-indigo-500/90 text-white font-semibold text-sm animate-slide-up';
            notif.textContent = mensaje;
            document.body.appendChild(notif);
            setTimeout(() => notif.remove(), 3000);
        },

        mostrarGanador(tipo, ganador) {
            const esPropio = this.cartones.some(c => c.id === ganador.carton_id);
            this.ganadorNombre = ganador.nombre || 'Jugador';
            
            if (esPropio) {
                this.winType = tipo === 'linea' ? '¡LÍNEA!' : '¡BINGO!';
                this.winAmount = ganador.premio;
                this.showWinModal = true;
                this.crearConfetti();
            } else {
                this.mostrarNotificacion(`🎉 ${this.ganadorNombre} ganó ${tipo === 'linea' ? 'LÍNEA' : 'BINGO'}! Premio: $${Number(ganador.premio).toLocaleString()}`);
            }
        },

        async multiplicarCarton(cartonId, multiplicador) {
            if (this.multiplicandoCarton) return;
            if (this.estado !== 'esperando') {
                this.mostrarNotificacion('🔒 Las apuestas están cerradas', 'alerta');
                return;
            }
            
            this.multiplicandoCarton = cartonId;
            
            try {
                const response = await fetch(`/multiplicar/${this.juegoId}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ multiplicador, carton_id: cartonId })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.acumulado = data.acumulado;
                    this.premioLinea = data.premio_linea;
                    this.premioBingo = data.premio_bingo;
                    if (data.balance !== undefined) {
                        this.balance = data.balance;
                    }
                    
                    this.cartones = this.cartones.map(c => {
                        if (c.id === cartonId) {
                            return { ...c, apuesta: data.nueva_apuesta };
                        }
                        return c;
                    });
                } else if (data.error) {
                    if (data.apuestas_cerradas) {
                        this.mostrarNotificacion('🔒 Las apuestas están cerradas', 'alerta');
                    }
                }
            } catch (error) {
                console.error('Error:', error);
            } finally {
                this.multiplicandoCarton = null;
            }
        },

        async ajustarApuesta(cartonId, ajuste) {
            if (this.ajustandoCarton) return;
            if (this.estado !== 'esperando') {
                this.mostrarNotificacion('🔒 Las apuestas están cerradas', 'alerta');
                return;
            }
            
            const carton = this.cartones.find(c => c.id === cartonId);
            if (ajuste < 0 && carton.apuesta <= 1) return;
            
            this.ajustandoCarton = cartonId;
            
            try {
                const response = await fetch(`/ajustar-apuesta/${this.juegoId}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    },
                    body: JSON.stringify({ ajuste, carton_id: cartonId })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.acumulado = data.acumulado;
                    this.premioLinea = data.premio_linea;
                    this.premioBingo = data.premio_bingo;
                    if (data.balance !== undefined) {
                        this.balance = data.balance;
                    }
                    
                    this.cartones = this.cartones.map(c => {
                        if (c.id === cartonId) {
                            return { ...c, apuesta: data.nueva_apuesta };
                        }
                        return c;
                    });
                } else if (data.error) {
                    if (data.apuestas_cerradas) {
                        this.mostrarNotificacion('🔒 Las apuestas están cerradas', 'alerta');
                    }
                }
            } catch (error) {
                console.error('Error:', error);
            } finally {
                this.ajustandoCarton = null;
            }
        },

        async toggleSorteoAuto() {
            if (this.sorteoAutoActivo) {
                if (!this.esControlador) return;
                await this.detenerSorteoAuto();
            } else {
                await this.iniciarSorteoAuto();
            }
        },

        async iniciarSorteoAuto() {
            try {
                const response = await fetch(`/sorteo-auto/iniciar/${this.juegoId}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.sorteoAutoActivo = true;
                    this.esControlador = true;
                    this.controladorNombre = this.nombreJugador;
                    this.sorteoEnVivo = true;
                    this.estado = 'en_curso';
                    this.startAutoPlay();
                }
            } catch (error) {
                console.error('Error iniciando sorteo auto:', error);
            }
        },

        async detenerSorteoAuto() {
            try {
                const response = await fetch(`/sorteo-auto/detener/${this.juegoId}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    this.sorteoAutoActivo = false;
                    this.esControlador = false;
                    this.stopAutoPlay();
                }
            } catch (error) {
                console.error('Error deteniendo sorteo auto:', error);
            }
        },

        startAutoPlay() {
            if (this.autoPlayInterval) return;
            
            this.autoPlayInterval = setInterval(() => {
                if (!this.isDrawing && this.estado !== 'finalizado' && this.esControlador) {
                    this.sortearBola();
                }
            }, 4500);
        },

        stopAutoPlay() {
            if (this.autoPlayInterval) {
                clearInterval(this.autoPlayInterval);
                this.autoPlayInterval = null;
            }
        },

        async sortearBola() {
            if (this.isDrawing || this.estado === 'finalizado') return;
            
            this.isDrawing = true;
            
            try {
                const response = await fetch(`/sortear/${this.juegoId}`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content,
                        'Accept': 'application/json'
                    }
                });
                
                const data = await response.json();
                
                if (data.error) {
                    this.isDrawing = false;
                    if (data.redirect) {
                        window.location.href = data.redirect;
                        return;
                    }
                    if (data.finalizado || data.limite_alcanzado) {
                        this.estado = 'finalizado';
                        this.autoPlay = false;
                        if (data.limite_alcanzado) {
                            this.showGameOverModal = true;
                        }
                    }
                    return;
                }
                
                setTimeout(() => {
                    this.ultimaBola = data.bola;
                    this.bolasSorteadas = data.bolas_sorteadas;
                    this.numerosSorteados.push(data.bola.numero);
                    this.historial.push(data.bola);
                    
                    // Actualizar estado a 'en_curso' cuando se sortea la primera bola
                    if (this.estado === 'esperando') {
                        this.estado = 'en_curso';
                        this.mostrarNotificacion('🔒 ¡Sorteo en curso! Apuestas cerradas', 'alerta');
                    }
                    
                    // Resaltar el número recién sorteado antes de marcarlo
                    this.numeroRecienSorteado = data.bola.numero;
                    
                    // Después de 1.5 segundos, marcar definitivamente y quitar resaltado
                    setTimeout(() => {
                        this.marcarEnCartones(data.bola.numero);
                        this.numeroRecienSorteado = null;
                    }, 1500);
                    
                    // Actualizar balance si viene en la respuesta
                    if (data.balance !== undefined) {
                        this.balance = data.balance;
                    }

                    if (data.ganador_linea && data.ganador_linea.es_mio) {
                        this.showWin('linea', data.ganador_linea.premio);
                        this.ultimaGanancia = data.ganador_linea.premio;
                        setTimeout(() => this.ultimaGanancia = 0, 5000);
                    }
                    
                    if (data.ganador_bingo && data.ganador_bingo.es_mio) {
                        this.showWin('bingo', data.ganador_bingo.premio);
                        this.ultimaGanancia = data.ganador_bingo.premio;
                        setTimeout(() => this.ultimaGanancia = 0, 5000);
                        this.estado = 'finalizado';
                        this.sorteoAutoActivo = false;
                        this.stopAutoPlay();
                    } else if (data.ganador_bingo) {
                        this.estado = 'finalizado';
                        this.sorteoAutoActivo = false;
                        this.stopAutoPlay();
                    }

                    if (data.finalizado || data.limite_alcanzado) {
                        this.estado = 'finalizado';
                        this.sorteoAutoActivo = false;
                        this.stopAutoPlay();
                        
                        if (data.limite_alcanzado && !data.ganador_bingo) {
                            setTimeout(() => {
                                this.showGameOverModal = true;
                            }, 500);
                        }
                    }
                    
                    this.isDrawing = false;
                }, 1800);
                
            } catch (error) {
                console.error('Error:', error);
                this.isDrawing = false;
            }
        },

        marcarEnCartones(numero) {
            this.cartones.forEach(carton => {
                for (let letra of ['B', 'I', 'N', 'G', 'O']) {
                    if (carton.numeros[letra].includes(numero)) {
                        if (!carton.marcados.includes(numero)) {
                            carton.marcados.push(numero);
                        }
                    }
                }
                
                carton.tiene_linea = this.verificarLinea(carton);
                carton.tiene_bingo = this.verificarBingo(carton);
            });
        },

        verificarLinea(carton) {
            for (let fila = 0; fila < 5; fila++) {
                let lineaCompleta = true;
                for (let letra of ['B', 'I', 'N', 'G', 'O']) {
                    let numero = carton.numeros[letra][fila];
                    if (numero !== 0 && !carton.marcados.includes(numero)) {
                        lineaCompleta = false;
                        break;
                    }
                }
                if (lineaCompleta) return true;
            }
            return false;
        },

        verificarBingo(carton) {
            for (let letra of ['B', 'I', 'N', 'G', 'O']) {
                for (let numero of carton.numeros[letra]) {
                    if (numero !== 0 && !carton.marcados.includes(numero)) {
                        return false;
                    }
                }
            }
            return true;
        },

        showWin(type, amount) {
            this.winType = type;
            this.winAmount = amount;
            this.showWinModal = true;
            
            // Lanzar confeti para bingo (más intenso) o línea (más suave)
            if (type === 'bingo') {
                this.launchConfetti(150, 8000);
            } else {
                this.launchConfetti(50, 4000);
            }
        },

        launchConfetti(count, duration) {
            const container = this.$refs.confettiContainer;
            const colors = ['#fbbf24', '#f59e0b', '#ef4444', '#22c55e', '#3b82f6', '#a855f7', '#ec4899', '#ffffff'];
            const shapes = ['square', 'circle', 'triangle'];
            
            for (let i = 0; i < count; i++) {
                setTimeout(() => {
                    const confetti = document.createElement('div');
                    const color = colors[Math.floor(Math.random() * colors.length)];
                    const shape = shapes[Math.floor(Math.random() * shapes.length)];
                    const size = Math.random() * 10 + 8;
                    const startX = Math.random() * 100;
                    const drift = (Math.random() - 0.5) * 200;
                    const rotationSpeed = Math.random() * 3 + 1;
                    const fallDuration = Math.random() * 3 + 3;
                    
                    confetti.className = 'confetti-piece';
                    confetti.style.cssText = `
                        position: absolute;
                        top: -20px;
                        left: ${startX}%;
                        width: ${size}px;
                        height: ${size}px;
                        background: ${color};
                        opacity: ${Math.random() * 0.5 + 0.5};
                        ${shape === 'circle' ? 'border-radius: 50%;' : ''}
                        ${shape === 'triangle' ? 'clip-path: polygon(50% 0%, 0% 100%, 100% 100%);' : ''}
                        animation: confetti-fall ${fallDuration}s linear forwards;
                        --drift: ${drift}px;
                        --rotation: ${rotationSpeed * 360}deg;
                    `;
                    
                    container.appendChild(confetti);
                    
                    setTimeout(() => confetti.remove(), fallDuration * 1000);
                }, Math.random() * (duration / 2));
            }
        },

        stopConfetti() {
            const container = this.$refs.confettiContainer;
            if (container) {
                container.innerHTML = '';
            }
        },

        isNumerSorteado(n) {
            return this.numerosSorteados.includes(n);
        },

        getNumerosPorLetra(letra) {
            const rangos = {
                'B': Array.from({length: 15}, (_, i) => i + 1),
                'I': Array.from({length: 15}, (_, i) => i + 16),
                'N': Array.from({length: 15}, (_, i) => i + 31),
                'G': Array.from({length: 15}, (_, i) => i + 46),
                'O': Array.from({length: 15}, (_, i) => i + 61)
            };
            return rangos[letra];
        },

        getColorLetra(letra) {
            const colores = {
                'B': 'bg-blue-500',
                'I': 'bg-red-500',
                'N': 'bg-gray-400',
                'G': 'bg-green-500',
                'O': 'bg-yellow-500'
            };
            return colores[letra];
        },

        getCartonCellClass(carton, letra, fila) {
            const numero = carton.numeros[letra][fila];
            
            // Espacio libre (centro)
            if (numero === 0) {
                return 'bg-gradient-to-br from-yellow-400 to-orange-500 text-white marked';
            }
            
            // Número ya marcado
            if (carton.marcados.includes(numero)) {
                return 'bg-gradient-to-br from-green-400 to-green-600 text-white marked animate-number-pop';
            }
            
            // Número recién sorteado que está en este cartón (resaltado temporal)
            if (this.numeroRecienSorteado === numero) {
                return 'bg-gradient-to-br from-orange-400 to-red-500 text-white font-black animate-pulse ring-2 ring-orange-300 ring-offset-1 scale-110 z-10 shadow-lg shadow-orange-500/50';
            }
            
            return 'bg-gray-100 text-gray-800 hover:bg-gray-200';
        }
    }
}
</script>
@endpush
@endsection
